function [ajd_idio, x0] = get_ajd_idio(cds, ajd_common, used_dates, z0, y0)
% --------------------------------------------------------------------------------------------------
% Given the dynamics of (i) lambda_it and (ii) the common factor Y, extract the time series of
% parameters governing the dynamics of X_it = lambda_it - a_i * Y. Dates with no parameters
% available use the value from the most recent date.
% --------------------------------------------------------------------------------------------------
% cds           ... credit default swap structure
% ajd_common    ... dynamics of common factor
% used_dates    ... desired dates for parameter time series
% --------------------------------------------------------------------------------------------------
% sample call: get_ajd_idio(cdx_tv_yearly.portfolio(52), cdx_tv_yearly.AJD_common_factor, [732211; 732647; 732982]);
%              get_ajd_idio(cdx_tv_yearly.portfolio(52), cdx_tv_yearly.AJD_common_factor, [732211; 732647; 732982], cdx_tv_yearly.portfolio(52).x0, cdx_tv_yearly.y0);
% --------------------------------------------------------------------------------------------------

% Determine matching positions between CDX dates, CDS dates and desired dates
[trash, used_pos_cds, used_pos_ajd] = intersect_sorted(cds.AJD.dates, used_dates);
found_dates = cds.AJD.dates(used_pos_cds);
used_pos_cdx = find(logical(is_member_sorted_c(ajd_common.dates, found_dates)));
num_dates = length(used_dates);

% Calculate implied dynamics of idionsyncratic risk factor
x0 = zeros(num_dates, 1);
ajd_idio = struct('k', x0 , 'theta', x0 , 'sigma', x0 , 'L', x0 , 'mu', x0 , 'dates', used_dates);
if (isempty(used_pos_ajd))
    return;
end
ajd_idio.k(used_pos_ajd,:)        = cds.AJD.k(used_pos_cds);              
ajd_idio.theta(used_pos_ajd,:)    = cds.AJD.theta(used_pos_cds) -  ajd_common.theta(used_pos_cdx) .* cds.ai(used_pos_cds);
ajd_idio.sigma(used_pos_ajd,:)    = cds.AJD.sigma(used_pos_cds);
ajd_idio.L(used_pos_ajd,:)        = cds.AJD.L(used_pos_cds) - ajd_common.L(used_pos_cdx);
ajd_idio.mu(used_pos_ajd,:)       = cds.AJD.mu(used_pos_cds);
if (nargin > 3)
    x0(used_pos_ajd,:)            = z0(used_pos_cds) - cds.ai(used_pos_cds) .* y0(used_pos_cdx);
end

% Fill in missing data with most recently available
if (num_dates > length(found_dates))
    % Determine closest dates where parameters are available
    missing_pos = find(~logical(is_member_sorted_c(used_dates, found_dates)));
    missing_dates = used_dates(missing_pos);
    impute_pos = used_pos_ajd(fuzzy_datenum_match(missing_dates, found_dates));
    
    % Fill in missing data
    ajd_idio.k(missing_pos)     = ajd_idio.k(impute_pos);
    ajd_idio.theta(missing_pos) = ajd_idio.theta(impute_pos);
    ajd_idio.sigma(missing_pos) = ajd_idio.sigma(impute_pos);
    ajd_idio.L(missing_pos)     = ajd_idio.L(impute_pos);
    ajd_idio.mu(missing_pos)    = ajd_idio.mu(impute_pos);
    x0(missing_pos)             = x0(impute_pos);
end